/***************************************************************************
 * Copyright 2012 BMW Car IT GmbH
 * Copyright (C) 2012 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#ifndef __CONFIGURATION_H__
#define __CONFIGURATION_H__

#include "config.h"
#include <ilm_types.h>
#include <string>
#include <map>

//============================================================================
// default configuration for LayerManagementService
//============================================================================
#define DEFAULT_DISPLAY_NAME        ":0"
#define DEFAULT_DISPLAY_WIDTH       1280
#define DEFAULT_DISPLAY_HEIGHT      480

#define DEFAULT_DISPLAY_COLOR_DEPTH 16
#define DEFAULT_DISPLAY_BUFFER_PER_OUTPUT 2
#define DEFAULT_DISPLAY_DITHERING 1
#define DEFAULT_IGNORE_EVENT_ON_ALPHA0 false
#define DEFAULT_DISP_ID_FOR_INPUT -1
#define DEFAULT_PRIMARY_CSC_DISP_ID 1
#define DEFAULT_SECONDARY_CSC_DISP_ID 2

#define DEFAULT_REPAINT_WINDOW_PERIOD_MS 8
#define DEFAULT_USE_MULTI_SRC_BLIT 1

#define DEFAULT_LOG_LEVEL_CONSOLE   2
#define DEFAULT_LOG_LEVEL_FILE      0
#define DEFAULT_LOG_LEVEL_TRACE     2

#define DEFAULT_FILTER_KERNEL_SIZE  0

#define DEFAULT_PLUGIN_PATH         CMAKE_INSTALL_PREFIX"/lib/layermanager"
#define DEFAULT_INPUTDEVFILEPATH "/etc/layermanager/device-configuration.conf"

//============================================================================
// configuration class
//============================================================================
class Configuration
{
public:
    Configuration(int argc, char** argv);

    std::string getDisplayName();
    int getDisplayWidth();
    int getDisplayHeight();

    int getDisplayForInput();

    int getDisplayColorDepth();
    int getDisplayBufferCnt();
    bool getDisplayDithering();
    int getFilterKernelSize();
    void disableFilterBlit();

    int getScreenColorDepth(unsigned int screen);
    int getScreenBufferCnt(unsigned int screen);
    bool getScreenDithering(unsigned int screen);

    bool getIgnoreEventOnAlpha0();

    int getLogLevelConsole();
    int getLogLevelFile();
    int getLogLevelTrace();

    std::string getPluginPath();

    void logAllSettings();
    std::string getInputDevConfigPath();
    void setInputDevConfigPath(std::string devconfigpath);
    bool setDisplayIDmapping(unsigned int HDMI,
                             unsigned int LVDS0,
                             unsigned int LVDS1,
                             unsigned int PARALLEL);

    bool setDisplayIDmapping(unsigned int *pDisplayIdArray,
                             unsigned int length);

    bool setCSCValues(unsigned int screenID, ilmCSCProperties* pcscProp);
    ilmCSCProperties* getCSCValues(unsigned int screenID);

    bool setCSCValues(unsigned int screenID,
                      ilmCSCProperties* pcscProp,
                      ilmUSERSpecificCSCvalues* pusercscProp);
    ilmCSCProperties* getCSCValues(unsigned int screenID,
                      ilmUSERSpecificCSCvalues* pusercscProp);

    int getHdmIid() const
    {
        return mDisplayIds[ILM_DISPLAY_HDMI];
    }

    int getLvds0id() const
    {
        return mDisplayIds[ILM_DISPLAY_LVDS0];
    }

    int getLvds1id() const
    {
        return mDisplayIds[ILM_DISPLAY_LVDS1];
    }

    int getParallelid() const
    {
        return mDisplayIds[ILM_DISPLAY_PARALLEL];
    }

    int getDsi0id() const
    {
        return mDisplayIds[ILM_DISPLAY_DSI0];
    }

    int getVga0id() const
    {
        return mDisplayIds[ILM_DISPLAY_VGA0];
    }

    int getVga1id() const
    {
        return mDisplayIds[ILM_DISPLAY_VGA1];
    }

    int getVga2id() const
    {
        return mDisplayIds[ILM_DISPLAY_VGA2];
    }

    int getVga3id() const
    {
        return mDisplayIds[ILM_DISPLAY_VGA3];
    }

    int getPrimaryCSCdisplayID() const
    {
        return mPrimaryCSCdisplayID;
    }

    void setPrimaryCSCdisplayID(int primaryCsCdisplayId)
    {
        mPrimaryCSCdisplayID = primaryCsCdisplayId;
    }

    int getSecondaryCSCdisplayID() const
    {
        return mSecondaryCSCdisplayID;
    }

    void setSecondaryCSCdisplayID(int secondaryCsCdisplayId)
    {
        mSecondaryCSCdisplayID = secondaryCsCdisplayId;
    }

    void setRepaintWindowPeriodMS(int repaintWindowPeriodMS)
    {
        mRepaintWindowPeriodMS = repaintWindowPeriodMS;
    }

    int getRepaintWindowPeriodMS() const
    {
       return mRepaintWindowPeriodMS;
    }

    void setEnableMultiSrcBlit(int enableMultiSrcBlit)
    {
        mEnableMultiSrcBlit = enableMultiSrcBlit;
    }

    int getEnableMultiSrcBlit() const
    {
        return mEnableMultiSrcBlit;
    }

private:
    void processCommandLine(int argc, char** argv);
    void processBuildFlags();
    void processEnvironmentVariables();

private:
    // command line arguments
    std::string mCommandLine;
    int mDisplayWidth;
    int mDisplayHeight;
    int mLogLevelConsole;
    int mLogLevelFile;
    int mLogLevelTrace;

    // environment
    std::string mDisplayName;
    std::string mPluginPath;

    /*configuration for display output*/
    int mDisplayColorDepth;
    int mDisplayBufferPerOutput;
    bool mDisplayDithering;
    int mFilterKernelSize;
    /*configuration of ignoring event on transparent pixels*/
    bool mIgnoreEventOnAlpha0;
    // build flags
    const BuildFlag* mpBuildFlags;
    const int mBuildFlagCount;
    bool mCustomConfig;
    int mDisplayIdForInput;
    std::string mInputDevConfigPath;
    /*configuration for display id mapping*/
    int mDisplayIds[ILM_DISPLAY_MAX_NUM];

    /*screen ids for csc*/
    int mPrimaryCSCdisplayID;
    int mSecondaryCSCdisplayID;

    int mRepaintWindowPeriodMS;

    int mEnableMultiSrcBlit;

    /*properties related to csc*/
    std::map<unsigned int, ilmCSCProperties*> m_CSCProperties;

    std::map<unsigned int, ilmUSERSpecificCSCvalues*> m_userspecificCSCvalues;

    bool containsNoDuplicates(unsigned int *pDisplayIdArray, unsigned int length);
    bool containsNoInvalidIds(unsigned int *pDisplayIdArray, unsigned int length);
    bool setDisplayIds(unsigned int *pDisplayIdArray, unsigned int length);
};

#endif // __CONFIGURATION_H__
